﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using UnityEngine;
using UnityEngine.Tilemaps;
using UnityEngine.XR;

//StreetsManager is a singleton class responsible for identifying and managing street names
public class StreetsManager : MonoBehaviour
{
    public Dictionary<Vector3Int, string> Streets { get; private set; } = new Dictionary<Vector3Int, string>();
    
    private List<string> _Prefixes = new List<string>();
    private List<string> _Suffixes = new List<string>();

    //Singleton
    private static StreetsManager _Instance;

    public static StreetsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<StreetsManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Initializes the StreetsManager by reading in the street prefixes and suffixes
    /// </summary>
    /// <returns>Did we initialize successfully?</returns>
    public bool Initialize()
    {
        try
        {
            //Read in the prefixes and suffixes
            _Prefixes = Utilities.ReadIxes("Streets/Prefixes");
            _Suffixes = Utilities.ReadIxes("Streets/Suffixes");
            return true;
        }

        catch(Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }


    /// <summary>
    /// Loops through the roads to identify all streets in the city
    /// </summary>
    public void IdentifyStreets()
    {
        #region Horizontal Pass
        //Let's loop through all roads horizontally
        for (int y = 0; y < GameManager.Instance.RoadsTilemap.size.y; y++)
        {
            for (int x = 0; x < GameManager.Instance.RoadsTilemap.size.x; x++)
            {
                Vector3Int thisTilePos = new Vector3Int(x, y, Constants.TilemapZPosition);
                TileBase thisTile = GameManager.Instance.RoadsTilemap.GetTile(thisTilePos);

                if (thisTile != null && ((Tile)thisTile).name == Constants.HorizontalRoadTileID)
                {
                    //It's a road tile, so add it
                    List<Vector3Int> thisStreetTilePositions = new List<Vector3Int>();
                    thisStreetTilePositions.Add(thisTilePos);

                    int thisStreetX = x + 1;
                    while (true)
                    {
                        //Now let's keep moving through horizontally to the right
                        if (thisStreetX < GameManager.Instance.RoadsTilemap.size.x)
                        {
                            Vector3Int thisStreetTilePos = new Vector3Int(thisStreetX, y, Constants.TilemapZPosition);
                            TileBase thisStreetTile = GameManager.Instance.RoadsTilemap.GetTile(thisStreetTilePos);

                            if (thisStreetTile != null && ((Tile)thisStreetTile).name == Constants.HorizontalRoadTileID)
                            {
                                //Still got a road tile here so it's part of this street
                                thisStreetTilePositions.Add(thisStreetTilePos);
                                thisStreetX++;
                                //Could draw a debug tile here to make sure
                            }

                            else
                            {
                                break;  //No road tile so it's the end of the street
                            }
                        }

                        else
                        {
                            break;  //This street ends on the boundary of the city
                        }
                    }

                    //We've got our street tiles, let's name it, add it to the list and move on
                    string thisStreetName = GenerateStreetName();
                    foreach (Vector3Int streetTilePos in thisStreetTilePositions)
                    {
                        Streets[streetTilePos] = thisStreetName;
                    }

                    x += (thisStreetX - x - 1);
                }
            }
        }
        #endregion

        #region Vertical Pass
        //Let's loop through all roads vertically
        for (int x = 0; x < GameManager.Instance.RoadsTilemap.size.x; x++)
        {
            for (int y = 0; y < GameManager.Instance.RoadsTilemap.size.y; y++)
            {
                Vector3Int thisTilePos = new Vector3Int(x, y, Constants.TilemapZPosition);
                TileBase thisTile = GameManager.Instance.RoadsTilemap.GetTile(thisTilePos);

                if (thisTile != null && ((Tile)thisTile).name == Constants.VerticalRoadTileID)
                {
                    //It's a road tile, so add it
                    List<Vector3Int> thisStreetTilePositions = new List<Vector3Int>();
                    thisStreetTilePositions.Add(thisTilePos);

                    int thisStreetY = y + 1;
                    while (true)
                    {
                        //Now let's keep moving through vertically to the top
                        if (thisStreetY < GameManager.Instance.RoadsTilemap.size.y)
                        {
                            Vector3Int thisStreetTilePos = new Vector3Int(x, thisStreetY, Constants.TilemapZPosition);
                            TileBase thisStreetTile = GameManager.Instance.RoadsTilemap.GetTile(thisStreetTilePos);

                            if (thisStreetTile != null && ((Tile)thisStreetTile).name == Constants.VerticalRoadTileID)
                            {
                                //Still got a road tile here so it's part of this street
                                thisStreetTilePositions.Add(thisStreetTilePos);
                                thisStreetY++;
                                //Could draw a debug tile here to make sure
                            }

                            else
                            {
                                break;  //No road tile so it's the end of the street
                            }
                        }

                        else
                        {
                            break;  //This street ends on the boundary of the city
                        }
                    }

                    //We've got our street tiles, let's name it, add it to the list and move on
                    string thisStreetName = GenerateStreetName();
                    foreach(Vector3Int streetTilePos in thisStreetTilePositions)
                    {
                        Streets[streetTilePos] = thisStreetName;
                    }

                    y += (thisStreetY - y - 1);
                }
            }
        }
        #endregion
    }

    /// <summary>
    /// Gets the name of the current street the player is positioned on
    /// </summary>
    /// <returns>The name of the street if valid, or string.Empty if not</returns>
    public string GetCurrentStreetName()
    {
        Vector3Int tilePos = GameManager.Instance.RoadsTilemap.WorldToCell(GameManager.Instance.PlayerCarGameObject.transform.position);

        if(Streets.ContainsKey(tilePos))
        {
            return Streets[tilePos];
        }

        else
        {
            return string.Empty;
        }
    }

    /// <summary>
    /// Utilizes Randomizer to return a random street name by combining a random prefix and suffix
    /// </summary>
    /// <returns>The generated street name</returns>
    private string GenerateStreetName()
    {
        Randomizer.Regenerate();
        int prefixIndex = Randomizer.RNG.Next(0, _Prefixes.Count - 1);
        Randomizer.Regenerate();
        int suffixIndex = Randomizer.RNG.Next(0, _Suffixes.Count - 1);
        name = _Prefixes[prefixIndex] + " " + _Suffixes[suffixIndex];
        return name;
    }
}
